<?php
/*
Plugin Name: LoadClient
Description: Verbindung zum PHPLABOR Repository zur Installation und Aktualisierung von Plugins und Themes - INSTALLATION NUR WENN ENTSPRECHENDE LIZENZ VORLIEGT
Version: 2.3.0
Author: PHPLABOR
*/

if (!defined('ABSPATH')) exit;

/**
 * Deaktiviert Cache für Admin-Seiten, um sicherzustellen, dass Änderungen sofort sichtbar werden.
 */
function loadclient_disable_cache() {
    nocache_headers();
    header("Cache-Control: no-store, no-cache, must-revalidate, max-age=0");
    header("Cache-Control: post-check=0, pre-check=0", false);
    header("Pragma: no-cache");
}
add_action('admin_init', 'loadclient_disable_cache');

class LoadClient {

    private $host_api = 'https://repo.phplabor.de/wp-content/plugins/phplabor-host/api.php';
    private $opt_license = 'phplabor_repo_license';
    private $opt_token   = 'phplabor_repo_token';

    public function __construct() {
        add_action('admin_menu', [$this, 'menu']);
		add_action('admin_menu', [$this, 'remove_duplicate_loadclient_submenu'], 999);
        register_activation_hook(__FILE__, [$this, 'install_log']);
    }

	public function remove_duplicate_loadclient_submenu() {
		remove_submenu_page('loadclient-pages', 'loadclient-pages');
	}

    public function menu() {
        add_menu_page('LoadClient', 'LoadClient', 'manage_options', 'loadclient-pages', [$this, 'show_license'], 'dashicons-download', 9999);
		add_submenu_page('loadclient-pages', 'Lizenz', 'Lizenz', 'manage_options', 'loadclient-license', [$this, 'show_license']);
        add_submenu_page('loadclient-pages', 'Plugins', 'Plugins', 'manage_options', 'loadclient-plugins', [$this, 'show_plugins']);
        add_submenu_page('loadclient-pages', 'Themes', 'Themes', 'manage_options', 'loadclient-themes', [$this, 'show_themes']);
        add_submenu_page('loadclient-pages', 'Protokoll', 'Protokoll', 'manage_options', 'loadclient-log', [$this, 'show_log']);
    }

    public function install_log() {
        global $wpdb;
        $wpdb->query("CREATE TABLE IF NOT EXISTS {$wpdb->prefix}phplabor_repo_log (
            id BIGINT AUTO_INCREMENT PRIMARY KEY,
            time DATETIME DEFAULT CURRENT_TIMESTAMP,
            event_type VARCHAR(50),
            message TEXT
        ) " . $wpdb->get_charset_collate());
    }

    private function get_license() {
        return [
            'key'   => get_option($this->opt_license),
            'token' => get_option($this->opt_token)
        ];
    }

    /**
     * Führt über die API-Aktion "check" eine Lizenzprüfung durch.
     * Liefert true zurück, wenn der Lizenzschlüssel als aktiv gilt.
     */
    private function check_license() {
        $lic = $this->get_license();
        if (empty($lic['key'])) {
            return false;
        }
        $check_url = add_query_arg([
            'action'       => 'check',
            'license_key'  => $lic['key']
        ], $this->host_api);
        $response = wp_remote_get($check_url, ['timeout' => 15]);
        if (is_wp_error($response)) {
            return false;
        }
        $data = json_decode(wp_remote_retrieve_body($response), true);
        if (!empty($data['status']) && $data['status'] === 'active') {
            return true;
        }
        return false;
    }

    private function fetch_data($type = null) {
        $lic = $this->get_license();
        if (!$lic['key'] || !$lic['token']) return [];

        $url = add_query_arg([
            'action'       => 'plugins',
            'license_key'  => $lic['key'],
            'client_token' => $lic['token']
        ], $this->host_api);

        $response = wp_remote_get($url, ['timeout' => 15]);
        $data = json_decode(wp_remote_retrieve_body($response), true);

        if (!is_array($data)) return [];

        return $type ? array_filter($data, fn($e) => $e['type'] === $type) : $data;
    }

    public function show_plugins() {
        echo '<div class="wrap"><h1>Plugins</h1>';
        $this->show_table('plugin');
        echo '</div>';
    }

    public function show_themes() {
        echo '<div class="wrap"><h1>Themes</h1>';
        $this->show_table('theme');
        echo '</div>';
    }

    private function show_table($type) {
        $list = $this->fetch_data($type);
        if (empty($list)) {
            echo '<p>Keine Einträge verfügbar, Lizenz nicht aktiv oder nicht im Installationsmodus.<br>Bitte dieses Plugin nicht ohne Rücksprache deinstallieren.</p>';
            return;
        }

        // Prüfe, ob die Lizenz aktiv ist (über API "check")
        $license_active = $this->check_license();

        echo '<table class="widefat striped"><thead><tr>
              <th>Name</th><th>Version</th><th>Beschreibung</th><th>Aktion</th>
              </tr></thead><tbody>';

        foreach ($list as $e) {
            $slug = sanitize_title($e['name']);
            $installed_version = null;
            $button_text = 'Installieren';

            if ($type === 'plugin') {
                foreach (get_plugins() as $file => $data) {
                    if (stripos($file, $slug) !== false) {
                        $installed_version = $data['Version'];
                        break;
                    }
                }
            } else {
                foreach (wp_get_themes() as $theme_slug => $theme_data) {
                    if ($slug === $theme_slug || sanitize_title($theme_data->get('Name')) === $slug) {
                        $installed_version = $theme_data->get('Version');
                        break;
                    }
                }
            }

            if ($installed_version && version_compare($e['version'], $installed_version, '>')) {
                $button_text = 'Update';
            }

            // Der Client erhält als Identifier die Datenbank-ID
            $install_url = admin_url("admin.php?page=loadclient-{$type}s&install=" . urlencode($e['id']) . "&type={$type}");
            if ($license_active) {
                $action_html = "<a href='" . esc_url($install_url) . "' class='button'>{$button_text}</a>";
            } else {
                $action_html = "<span class='button' style='background:#ccc;cursor:not-allowed;'>Lizenz nicht aktiv</span>";
            }

            echo "<tr>
                <td>{$e['name']}</td>
                <td>{$e['version']}</td>
                <td>{$e['description']}</td>
                <td>{$action_html}</td>
            </tr>";
        }

        echo '</tbody></table>';

        if (isset($_GET['install']) && isset($_GET['type'])) {
            if (!$license_active) {
                echo '<div class="error"><p>Installation/Update ist deaktiviert, da die Lizenz nicht aktiv ist.</p></div>';
            } else {
                $this->install_from_zip(sanitize_text_field($_GET['install']), sanitize_text_field($_GET['type']));
            }
        }
    }

    private function install_from_zip($id, $type) {
        $lic = $this->get_license();
        if (!$lic['key'] || !$lic['token']) return;

        // Erzeuge die API-URL zum Download anhand der übergebenen ID (als GET-Parameter "install")
        $url = add_query_arg([
            'action'       => 'download',
            'license_key'  => $lic['key'],
            'client_token' => $lic['token'],
            'install'      => $id
        ], $this->host_api);
        error_log("[LoadClient] Download URL: " . $url);

        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        WP_Filesystem();

        $tmp_file = download_url($url);
        if (is_wp_error($tmp_file)) {
            $err = $tmp_file->get_error_message();
            error_log("[LoadClient] Download fehlgeschlagen: " . $err);
            $this->log('Fehler', 'Download fehlgeschlagen: ' . $err);
            echo '<div class="error"><p>Download fehlgeschlagen: ' . $err . '</p></div>';
            return;
        }

        if (!file_exists($tmp_file)) {
            error_log("[LoadClient] ERROR: Temporäre Datei existiert nicht!");
        } else {
            $filesize = filesize($tmp_file);
            error_log("[LoadClient] Temporäre Datei existiert. Größe: " . $filesize . " Byte");
        }

        $skin = new Automatic_Upgrader_Skin();
        if ($type === 'theme') {
            $upgrader = new Theme_Upgrader($skin);
            $destination = get_theme_root(); // Zielverzeichnis für Themes
        } else {
            $upgrader = new Plugin_Upgrader($skin);
            $destination = WP_PLUGIN_DIR;   // Zielverzeichnis für Plugins
        }
        error_log("[LoadClient] Starte Installationsprozess mit " . get_class($upgrader));

        // Verwenden der run()-Methode mit Force-Optionen, um ein Update zu erzwingen
        $options = [
            'package'           => $tmp_file,
            'destination'       => $destination,
            'clear_destination' => true,
            'clear_working'     => true,
            'hook_extra'        => ['type' => $type, 'action' => 'update']
        ];

        ob_start();
        $result = $upgrader->run($options);
        $upgrader_output = ob_get_clean();
        error_log("[LoadClient] Upgrader-Ausgabe: " . $upgrader_output);
        error_log("[LoadClient] Ergebnis der Upgrader run(): " . var_export($result, true));

        unlink($tmp_file);

        if (is_null($result)) {
            error_log("[LoadClient] Upgrader run() gibt NULL zurück - Erfolg angenommen.");
            $result = true;
        }

        if (is_wp_error($result)) {
            $this->log('Fehler', 'Installation fehlgeschlagen: ' . $result->get_error_message());
            echo '<div class="error"><p>Installation fehlgeschlagen: ' . $result->get_error_message() . '</p></div>';
        } else {
            $this->log('Install', ucfirst($type) . ' erfolgreich installiert oder aktualisiert.');
            echo '<div class="updated"><p>' . ucfirst($type) . ' erfolgreich installiert oder aktualisiert. Bitte aktivieren Sie das Plugin/Theme.</p></div>';
        }

        echo '<p><a href="' . admin_url('admin.php?page=loadclient-' . $type . 's') . '">Zurück zur Liste</a></p>';
        exit;
    }

    public function show_license() {
        if (!empty($_POST['license_key'])) {
            $key = sanitize_text_field($_POST['license_key']);
            $response = wp_remote_get(add_query_arg([
                'action' => 'activate',
                'license_key' => $key
            ], $this->host_api));
            $result = json_decode(wp_remote_retrieve_body($response), true);
            if (!empty($result['client_token'])) {
                update_option($this->opt_license, $key);
                update_option($this->opt_token, $result['client_token']);
                $this->log('Lizenz', 'Aktivierung erfolgreich');
                echo '<div class="updated"><p>Lizenz aktiviert.</p></div>';
            } else {
                $this->log('Lizenz', 'Aktivierung fehlgeschlagen');
                echo '<div class="error"><p>Lizenz ungültig oder gesperrt.</p></div>';
            }
        }

        // Lizenzstatus über die API "check" abrufen
        $lic_active = false;
        $saved_key = get_option($this->opt_license);
        if (!empty($saved_key)) {
            $check_url = add_query_arg([
                'action' => 'check',
                'license_key' => $saved_key
            ], $this->host_api);
            $check_response = wp_remote_get($check_url, ['timeout' => 15]);
            $check_result = json_decode(wp_remote_retrieve_body($check_response), true);
            if (!empty($check_result['status']) && $check_result['status'] === 'active') {
                $lic_active = true;
            }
        }
        $status = $lic_active ? 'Aktiv' : 'Nicht aktiv';

        echo '<div class="wrap"><h1>Lizenz</h1>';
        echo "<p>Status: <strong>{$status}</strong></p>";
        echo '<form method="post"><input type="text" name="license_key" placeholder="Lizenzschlüssel" value="" size="50">';
        submit_button('Aktivieren / Ändern');
        echo '</form></div>';
    }

    public function show_log() {
        global $wpdb;
        $logs = $wpdb->get_results("SELECT * FROM {$wpdb->prefix}phplabor_repo_log ORDER BY time DESC LIMIT 100");
        echo '<div class="wrap"><h1>Protokoll</h1><table class="widefat striped"><thead><tr><th>Zeit</th><th>Typ</th><th>Nachricht</th></tr></thead><tbody>';
        foreach ($logs as $log) {
            echo "<tr><td>{$log->time}</td><td>{$log->event_type}</td><td>{$log->message}</td></tr>";
        }
        echo '</tbody></table></div>';
    }

    private function log($type, $msg) {
        global $wpdb;
        $wpdb->insert($wpdb->prefix . 'phplabor_repo_log', ['event_type' => $type, 'message' => $msg]);
    }
}

new LoadClient();

register_uninstall_hook(__FILE__, 'loadclient_uninstall');
function loadclient_uninstall() {
    global $wpdb;
    $wpdb->query("DROP TABLE IF EXISTS {$wpdb->prefix}phplabor_repo_log");
    delete_option('phplabor_repo_license');
    delete_option('phplabor_repo_token');
}
